using System;
using System.Data;
using System.Collections;
using System.Diagnostics;
using System.Reflection;
using gov.va.med.vbecs.DAL;
using gov.va.med.vbecs.Common;
using gov.va.med.vbecs.ExceptionManagement;
using TABLE = gov.va.med.vbecs.Common.VbecsTables;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;

namespace gov.va.med.vbecs.BOL
{

	#region Header

    ///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
    ///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
    ///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
    ///<Developers>
    ///	<Developer>John Vrooland</Developer>
    ///</Developers>
    ///<SiteName>Hines OIFO</SiteName>
    ///<CreationDate>1/31/2003</CreationDate>
    ///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	/// <summary>
	/// WorkLoad provides the lookup and association of LMIP/NLT
	/// codes to VBECS processes.
	/// </summary>

		#endregion

    public class WorkLoad : BaseBusinessObject, IRecordStatusCode
    {		
		#region Constants
		/// <summary>
		/// Work Load Minimum Columns
		/// </summary>
		public const string WORKLOAD_MINCOLUMNS = "AssignedDate,Cost,DivisionCode,LastUpdateDate,LastUpdateUser,RowVersion,UnAssignedDate,WorkloadCode,WorkloadProcedure,WorkloadProcessGuid";
#if NUNIT
		/// <summary>
		/// Work Load Minimum Columns
		/// </summary>
		public const string WORKLOADMINCOLUMNS = WORKLOAD_MINCOLUMNS;
#endif

		#endregion

		#region Class Variables
		DataRow _workLoadRow;
        ArrayList _cptCodes = new ArrayList();
		private Common.RecordStatusCode _recordStatusCode;
		#endregion

		#region Constructors



		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/10/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1640"> 
		///		<ExpectedInput>Valid datarow</ExpectedInput>
		///		<ExpectedOutput>Valid workload object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1641"> 
		///		<ExpectedInput>Invalid datarow</ExpectedInput>
		///		<ExpectedOutput>Argument exception</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
        /// <summary>
        /// Public constructor requiring a datarow argument.
        /// </summary>
        /// <param name="workLoadRow">DataRow containing WorkLoad columns</param>
        public WorkLoad(DataRow workLoadRow)
        {
            if (!VerifyMinColumnsInDataTable(WORKLOAD_MINCOLUMNS.Split(','), workLoadRow.Table))
            {
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InsufficientColumns(
					MethodBase.GetCurrentMethod().Name).ResString);
            }
            _workLoadRow = workLoadRow;
			this.RecordStatusCode = Common.Utility.GetRecordStatusCodeFromString(_workLoadRow[TABLE.WorkloadEvent.RecordStatusCode].ToString());
			if (this.WorkloadProcessGuid == Guid.Empty) 
			{
				_workLoadRow[TABLE.WorkloadProcess.WorkloadProcessGuid] = Guid.NewGuid();
			}
            this._workLoadRow.Table.ColumnChanged += new DataColumnChangeEventHandler(this.OnColumnChanged);

			this.IsNew = false;
        }
        ///<Developers>
        ///	<Developer>John Vrooland</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>1/31/2003</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="1564"> 
        ///		<ExpectedInput>Valid datarows</ExpectedInput>
        ///		<ExpectedOutput>Valid workload object</ExpectedOutput>
        ///	</Case>
        ///
		///<Case type="1" testid ="1642"> 
		///		<ExpectedInput>Invalid datarows</ExpectedInput>
		///		<ExpectedOutput>Argument exception</ExpectedOutput>
		///	</Case>
		///
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
        /// <summary>
        /// Public constructor requiring DataRow and associated CPTCode arguments.
        /// </summary>
        /// <param name="workLoadRow">DataRow containing WorkLoad data</param>
        /// <param name="cptCodes">strings of associated CPTCodes</param>
        public WorkLoad(DataRow workLoadRow, DataRow[] cptCodes) : this(workLoadRow)
        {
			if (cptCodes != null && 
				cptCodes.Length > 0 && 
				cptCodes[0].Table.Columns.Contains(ARTIFICIAL.CPTCode_Text)) 
			{
				foreach ( DataRow dr in cptCodes )
				{
					_cptCodes.Add(dr[ARTIFICIAL.CPTCode_Text]);
				}
				this.IsDirty = false;
			}
			else 
			{
				string errorMessage = Common.StrRes.SysErrMsg.Common.InvalidPropertyOrMethod(
										Common.Utility.ParseStringwithSpaces(MethodBase.GetCurrentMethod().Name),
										"cptCodes argument").ResString;
				throw new ArgumentException(errorMessage);
			}

			this.IsNew = false;
        }
		#endregion

		#region Events
		///<Developers>
        ///	<Developer>John Vrooland</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>1/31/2003</CreationDate>
        ///
        /// <summary>
        /// Handler for detection of data modifications 
        /// responsible for setting dirty flag.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="eventArgs"></param>
        protected void OnColumnChanged(object sender, DataColumnChangeEventArgs eventArgs)
        {
            if (eventArgs.Column.ColumnName == TABLE.WorkloadProcess.AssignedDate || eventArgs.Column.ColumnName == TABLE.WorkloadProcess.UnAssignedDate) 
            {
                    this.IsDirty = true;
            } 
        }

		#endregion

		#region Methods

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/5/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3459"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Valid data table</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="1535"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Get the list of Vbecs processes
		/// </summary>
		/// <returns></returns>
		public static DataTable GetAllVbecsProcesses()
		{
			return DAL.WorkLoad.GetAllVbecsProcesses();
		}
		
		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/16/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6687"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6688"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// UpdateRecordStatusCode
		/// </summary>
		/// <param name="recordStatusCode">recordStatusCode</param>
		public void UpdateRecordStatusCode(Common.RecordStatusCode recordStatusCode)
		{
			_workLoadRow[TABLE.WorkloadProcess.RecordStatusCode] = 
				Common.Utility.GetRecordStatusCodeCharFromEnum(recordStatusCode);
		}


		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/10/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1643"> 
		///		<ExpectedInput>Datarow</ExpectedInput>
		///		<ExpectedOutput>Datarow</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1644"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
        /// <summary>
        /// Unused.
        /// </summary>
        /// <param name="dr"></param>
        /// <returns></returns>
        public override DataRow LoadDataRowFromThis(DataRow dr)
        {
            return dr;
        }

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/10/2003</CreationDate>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
        /// <summary>
        /// Unused.
        /// </summary>
        /// <param name="dr"></param>
        protected override void LoadFromDataRow(DataRow dr)
        {
        }
        ///<Developers>
        ///	<Developer>John Vrooland</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>1/31/2003</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="1565"> 
        ///		<ExpectedInput>Valid string of columns and datatable</ExpectedInput>
        ///		<ExpectedOutput>True</ExpectedOutput>
        ///	</Case>
        ///
		///<Case type="1" testid ="1648"> 
		///		<ExpectedInput>Invalid string of columns</ExpectedInput>
		///		<ExpectedOutput>False</ExpectedOutput>
		///	</Case>
		///
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
        /// <summary>
        /// Verify our data requirements are met.
        /// </summary>
        /// <param name="cols">Array of strings containing column names</param>
        /// <param name="dt">DataTable to verify presence of required columns</param>
        /// <returns></returns>
        public bool VerifyMinColumnsInDataTable(string[] cols, DataTable dt)
        {
            return Common.Utility.VerifyMinColumnsInDataTable(cols, dt);
        }


		#endregion

		#region Properties
		///<Developers>
        ///	<Developer>John Vrooland</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>1/31/2003</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="1566"> 
        ///		<ExpectedInput>None</ExpectedInput>
        ///		<ExpectedOutput>Workload code string</ExpectedOutput>
        ///	</Case>
		///	
		///<Case type="1" testid ="1604"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
        /// <summary>
        /// Gets or set the 10-character Workload code.
        /// </summary>
        public string WorkloadCode
        {
            get 
            {
                if ( _workLoadRow.IsNull(TABLE.WorkloadProcess.WorkloadCode) )
                {
                    return "";
                }
                return _workLoadRow[TABLE.WorkloadProcess.WorkloadCode].ToString();
            }
        }
        ///<Developers>
        ///	<Developer>John Vrooland</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>1/31/2003</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="1567"> 
        ///		<ExpectedInput>None</ExpectedInput>
        ///		<ExpectedOutput>Workload process GUID</ExpectedOutput>
        ///	</Case>
		///	
		///<Case type="1" testid ="1609"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
        /// <summary>
        /// Gets or sets the Guid of a WorloadProcess row.
        /// </summary>
        public System.Guid WorkloadProcessGuid
        {
            get 
            {
                if ( _workLoadRow.IsNull(TABLE.WorkloadProcess.WorkloadProcessGuid) )
                {
                    return System.Guid.Empty;
                }
                return (System.Guid)_workLoadRow[TABLE.WorkloadProcess.WorkloadProcessGuid];
            }
        }
        ///<Developers>
        ///	<Developer>John Vrooland</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>1/31/2003</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="1568"> 
        ///		<ExpectedInput>Valid dates</ExpectedInput>
        ///		<ExpectedOutput>Valid assigned date property</ExpectedOutput>
        ///	</Case>
		///	<Case type="1" testid ="1647"> 
		///		<ExpectedInput>Invalid date</ExpectedInput>
		///		<ExpectedOutput>Argument exception</ExpectedOutput>
		///	</Case>
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
        /// <summary>
        /// Gets or sets the AssignedDate value 
        /// </summary>
        public DateTime AssignedDate 
        {
            get 
            {
                if ( _workLoadRow.IsNull(TABLE.WorkloadProcess.AssignedDate) )
                {
                    return DateTime.MinValue;
                }
                return (DateTime)_workLoadRow[TABLE.WorkloadProcess.AssignedDate];
            }
            set 
            {
                if (value.CompareTo(DateTime.MinValue) < 1) 
                {
					string errorMessage = Common.StrRes.SysErrMsg.Common.InvalidPropertyOrMethod(
											Common.Utility.ParseStringwithSpaces(MethodBase.GetCurrentMethod().Name, true),
											value).ResString;
					throw new ArgumentException(errorMessage, "value");
                }
                _workLoadRow[TABLE.WorkloadProcess.AssignedDate] = value;
            }
        }

		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/26/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6560"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6561"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// The status of the record
		/// </summary>
		public Common.RecordStatusCode RecordStatusCode
		{
			get
			{
				return _recordStatusCode;
			}
			set
			{
				_recordStatusCode = value;
			}
		}
        ///<Developers>
        ///	<Developer>John Vrooland</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>1/31/2003</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="1569"> 
        ///		<ExpectedInput>Valid dates</ExpectedInput>
        ///		<ExpectedOutput>Valid assigned date</ExpectedOutput>
        ///	</Case>
        ///
		///<Case type="1" testid ="1649"> 
		///		<ExpectedInput>Invalid unassigned date (before assigned date)</ExpectedInput>
		///		<ExpectedOutput>Argument exception</ExpectedOutput>
		///	</Case>
		///
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
        /// <summary>
        /// Gets or set the UnAssignedDate value.
        /// Implements BR_29.06
        /// </summary>
        public DateTime UnAssignedDate
        {
            get 
            {
                if ( _workLoadRow.IsNull(TABLE.WorkloadProcess.UnAssignedDate) )
                {
                    return DateTime.MinValue;
                }
                return (DateTime)_workLoadRow[TABLE.WorkloadProcess.UnAssignedDate];
            }
            set 
            {
                if (value == DateTime.MinValue) 
                {
                    _workLoadRow[TABLE.WorkloadProcess.UnAssignedDate] = DBNull.Value;
                }
                else 
                {
                    if ( value.Date.CompareTo( this.AssignedDate.Date ) < 0 ) 
                    {
						string errorMessage = Common.StrRes.SysErrMsg.Common.CannotbeLessError("Unassigned date", "Assigned date").ResString;
						throw new ArgumentException(errorMessage);
                    }
                    _workLoadRow[TABLE.WorkloadProcess.UnAssignedDate] = value;
                }
            }
        }
        ///<Developers>
        ///	<Developer>John Vrooland</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>1/31/2003</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="1570"> 
        ///		<ExpectedInput>None</ExpectedInput>
        ///		<ExpectedOutput>Procedure description string</ExpectedOutput>
        ///	</Case>
		///	
		///<Case type="1" testid ="1603"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
        /// <summary>
        /// Gets the Procedure description text.
        /// </summary>
//        public string Procedure
//        {
//            get 
//            {
//                if ( _workLoadRow.IsNull(ARTIFICIAL.Procedure) )
//                {
//                    return "";
//                }
//                return _workLoadRow[ARTIFICIAL.Procedure].ToString();
//            }
//        }
		public string WorkloadProcedure
		{
			get 
			{
				if ( _workLoadRow.IsNull("WorkloadProcedure") )
				{
					return "";
				}
				return _workLoadRow["WorkloadProcedure"].ToString();
			}
		}

        ///<Developers>
        ///	<Developer>John Vrooland</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>1/31/2003</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="1571"> 
        ///		<ExpectedInput>NA</ExpectedInput>
        ///		<ExpectedOutput>ArrayList of CPTCodes</ExpectedOutput>
        ///	</Case>
		///	
		///<Case type="1" testid ="1509"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
        /// <summary>
        /// Gets the list of associated CPTCodes
        /// </summary>
        public ArrayList CPTCodes
        {
            get 
            {
                return _cptCodes;
            }
        }
        ///<Developers>
        ///	<Developer>John Vrooland</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>1/31/2003</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="1572"> 
        ///		<ExpectedInput>None</ExpectedInput>
        ///		<ExpectedOutput>Decimal cost value</ExpectedOutput>
        ///	</Case>
		///	
		///<Case type="1" testid ="1508"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
        /// <summary>
        /// Gets or sets the Cost associated with this procedure.
        /// Retrieved from VistA.
        /// </summary>
        public decimal Cost 
        {
            get 
            {
				if ( _workLoadRow.IsNull(ARTIFICIAL.Cost) )
                {
                    return 0.0m;
                }
                return (decimal)_workLoadRow[ARTIFICIAL.Cost];
            }
        }
		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4171"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>byte</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4172"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// WeightMultiplier
		/// </summary>
		public byte WeightMultiplier 
		{
			get 
			{
				if ( _workLoadRow.IsNull(TABLE.WorkloadEvent.WeightMultiplier) )
				{
					return 0;
				}
				return (byte) Convert.ToInt32(_workLoadRow[TABLE.WorkloadEvent.WeightMultiplier]);
			}
		}
        ///<Developers>
        ///	<Developer>John Vrooland</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>1/31/2003</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="1575"> 
        ///		<ExpectedInput>None</ExpectedInput>
        ///		<ExpectedOutput>Division code string</ExpectedOutput>
        ///	</Case>
		///	
		///<Case type="1" testid ="1531"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
        /// <summary>
        /// Gets or sets the division code.
        /// </summary>
        public string DivisionCode
        {
            get 
            {
                if ( _workLoadRow.IsNull(TABLE.WorkloadProcess.DivisionCode) )
                {
                    return "";
                }
                return _workLoadRow[TABLE.WorkloadProcess.DivisionCode].ToString();
            }
        }
        ///<Developers>
        ///	<Developer>John Vrooland</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>1/31/2003</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="1574"> 
        ///		<ExpectedInput>None</ExpectedInput>
        ///		<ExpectedOutput>Datetime - last update date</ExpectedOutput>
        ///	</Case>
		///	
		///<Case type="1" testid ="1536"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
        /// <summary>
        /// Gets the date and time this row last updated.
        /// </summary>
        public new DateTime LastUpdateDate
        {
            get 
            {
                if ( _workLoadRow.IsNull(TABLE.WorkloadProcess.LastUpdateDate) )
                {
                    return DateTime.MinValue;
                }
                return (DateTime)_workLoadRow[TABLE.WorkloadProcess.LastUpdateDate];
            }
        }

		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/26/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6558"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6559"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
        /// <summary>
        /// Gets the RowVersion value.
        /// </summary>
        public new byte[] RowVersion
        {
            get 
            {
                if ( _workLoadRow.IsNull(TABLE.WorkloadProcess.RowVersion) )
                {
                    return null;
                }
                return (byte[])_workLoadRow[TABLE.WorkloadProcess.RowVersion];
            }
        }
        ///<Developers>
        ///	<Developer>John Vrooland</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>1/31/2003</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="1573"> 
        ///		<ExpectedInput>None</ExpectedInput>
        ///		<ExpectedOutput>Last updating user ID</ExpectedOutput>
        ///	</Case>
		///	
		///<Case type="1" testid ="1583"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
        /// <summary>
        /// Gets the last updating user's id.
        /// </summary>
        public new string LastUpdateUser
        {
            get 
            {
                if ( _workLoadRow.IsNull(TABLE.WorkloadProcess.LastUpdateUser) )
                {
                    return "";
                }
                return _workLoadRow[TABLE.WorkloadProcess.LastUpdateUser].ToString();
            }
        }
        //
        internal DataRow WorkLoadRow
        {
            get
            {
                return _workLoadRow;
            }
        }

		#endregion
    }
}
